/*
 * Copyright (c) 2019 AlexIIL
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */
package alexiil.mc.lib.multipart.impl;

import java.util.List;

import javax.annotation.Nullable;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.rendering.data.v1.RenderAttachedBlockView;

import net.minecraft.block.Block;
import net.minecraft.block.BlockEntityProvider;
import net.minecraft.block.BlockState;
import net.minecraft.block.ShapeContext;
import net.minecraft.block.Waterloggable;
import net.minecraft.block.entity.BlockEntity;
import net.minecraft.block.entity.BlockEntityTicker;
import net.minecraft.block.entity.BlockEntityType;
import net.minecraft.client.MinecraftClient;
import net.minecraft.entity.Entity;
import net.minecraft.entity.ItemEntity;
import net.minecraft.entity.LivingEntity;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.fluid.Fluid;
import net.minecraft.fluid.FluidState;
import net.minecraft.fluid.Fluids;
import net.minecraft.item.ItemStack;
import net.minecraft.loot.context.LootContextParameterSet;
import net.minecraft.loot.context.LootContextParameters;
import net.minecraft.loot.context.LootContextTypes;
import net.minecraft.server.world.ServerWorld;
import net.minecraft.state.StateManager;
import net.minecraft.state.property.BooleanProperty;
import net.minecraft.state.property.EnumProperty;
import net.minecraft.state.property.IntProperty;
import net.minecraft.state.property.Properties;
import net.minecraft.util.ActionResult;
import net.minecraft.util.BlockMirror;
import net.minecraft.util.BlockRotation;
import net.minecraft.util.Hand;
import net.minecraft.util.ItemActionResult;
import net.minecraft.util.collection.DefaultedList;
import net.minecraft.util.hit.BlockHitResult;
import net.minecraft.util.hit.HitResult;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Box;
import net.minecraft.util.math.Direction;
import net.minecraft.util.math.DirectionTransformation;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.math.random.Random;
import net.minecraft.util.shape.VoxelShape;
import net.minecraft.util.shape.VoxelShapes;
import net.minecraft.world.BlockView;
import net.minecraft.world.GameRules;
import net.minecraft.world.World;
import net.minecraft.world.WorldAccess;
import net.minecraft.world.WorldView;
import net.minecraft.world.biome.Biome;

import alexiil.mc.lib.multipart.api.AbstractPart;
import alexiil.mc.lib.multipart.api.AbstractPart.ItemDropTarget;
import alexiil.mc.lib.multipart.api.PartLootParams;
import alexiil.mc.lib.multipart.api.SubdividedPart;
import alexiil.mc.lib.multipart.api.event.PartEventEntityCollide;
import alexiil.mc.lib.multipart.api.event.PartPrecipitationTickEvent;
import alexiil.mc.lib.multipart.api.event.PartRandomDisplayTickEvent;
import alexiil.mc.lib.multipart.api.event.PartRandomTickEvent;
import alexiil.mc.lib.multipart.api.event.PartScheduledTickEvent;
import alexiil.mc.lib.multipart.api.property.MultipartProperties;
import alexiil.mc.lib.multipart.impl.TransientPartIdentifier.IdAdditional;
import alexiil.mc.lib.multipart.impl.TransientPartIdentifier.IdSubPart;
import alexiil.mc.lib.multipart.impl.client.PartModelData;
import alexiil.mc.lib.multipart.mixin.api.IBlockCustomParticles;
import alexiil.mc.lib.multipart.mixin.api.IBlockDynamicCull;
import alexiil.mc.lib.multipart.mixin.api.IBlockMultipart;

public class MultipartBlock extends Block
    implements BlockEntityProvider, IBlockMultipart<TransientPartIdentifier>, Waterloggable, IBlockDynamicCull,
    IBlockCustomParticles {

    public static final IntProperty LUMINANCE = IntProperty.of("luminance", 0, 15);
    public static final BooleanProperty EMITS_REDSTONE = BooleanProperty.of("emits_redstone");

    /** BlockState-side transformation property used for detecting when the BlockState has been transformed without the
     * MultipartBlockEntity being involved.
     * <p>
     * The actual value of this property is meaningless. Only the difference between this and the block-entity's cached
     * transformation is meaningful. And even then, a difference only means that the block-entity should fire the
     * appropriate transformation events when it gets the chance. A part's orientation and transformation should
     * *always* be stored in the part itself.
     *
     * @see alexiil.mc.lib.multipart.api.event.PartTransformEvent */
    public static final EnumProperty<DirectionTransformation> TRANSFORMATION =
        EnumProperty.of("transformation", DirectionTransformation.class);

    public MultipartBlock(Settings settings) {
        super(settings);
        setDefaultState(
            getDefaultState()//
                .with(LUMINANCE, 0)//
                .with(EMITS_REDSTONE, false)//
                .with(Properties.WATERLOGGED, false)//
                .with(TRANSFORMATION, DirectionTransformation.IDENTITY)//
        );
    }

    // ###############
    //
    // Misc
    //
    // ###############

    @Override
    protected void appendProperties(StateManager.Builder<Block, BlockState> builder) {
        super.appendProperties(builder);
        builder.add(LUMINANCE, EMITS_REDSTONE, Properties.WATERLOGGED, TRANSFORMATION);
    }

    @Override
    public BlockEntity createBlockEntity(BlockPos pos, BlockState state) {
        return new MultipartBlockEntity(pos, state);
    }

    @Override
    public <T extends BlockEntity> BlockEntityTicker<T> getTicker(
        World world, BlockState state, BlockEntityType<T> type
    ) {
        return type == LibMultiPart.BLOCK_ENTITY ? (w, p, s, be) -> ((MultipartBlockEntity) be).tick() : null;
    }

    @Override
    public BlockState rotate(BlockState state, BlockRotation rotation) {
        return state.with(TRANSFORMATION, state.get(TRANSFORMATION).prepend(rotation.getDirectionTransformation()));
    }

    @Override
    public BlockState mirror(BlockState state, BlockMirror mirror) {
        return state.with(TRANSFORMATION, state.get(TRANSFORMATION).prepend(mirror.getDirectionTransformation()));
    }

    @Override
    public VoxelShape getCollisionShape(BlockState state, BlockView view, BlockPos pos, ShapeContext ctx) {
        if (view instanceof RenderAttachedBlockView renderView) {
            Object data = renderView.getBlockEntityRenderAttachment(pos);
            if (data instanceof PartModelData partData) {
                return partData.collisionShape;
            }
        }

        if (view.getBlockEntity(pos) instanceof MultipartBlockEntity container) {
            return container.container.getCollisionShape();
        }
        return VoxelShapes.empty();
    }

    @Override
    public VoxelShape getOutlineShape(BlockState state, BlockView view, BlockPos pos, ShapeContext ctx) {
        if (view instanceof RenderAttachedBlockView renderView) {
            Object data = renderView.getBlockEntityRenderAttachment(pos);
            if (data instanceof PartModelData partData) {
                return partData.outlineShape;
            }
        }

        if (view.getBlockEntity(pos) instanceof MultipartBlockEntity container) {
            return container.container.getOutlineShape();
        }
        return VoxelShapes.empty();
    }

    @Override
    public VoxelShape getCullingShape(BlockState state, BlockView view, BlockPos pos) {
        if (view instanceof RenderAttachedBlockView renderView) {
            Object data = renderView.getBlockEntityRenderAttachment(pos);
            if (data instanceof PartModelData partData) {
                return partData.cullingShape;
            }
        }

        if (view.getBlockEntity(pos) instanceof MultipartBlockEntity container) {
            return container.container.getCullingShape();
        }
        return super.getCullingShape(state, view, pos);
    }

    @Override
    public VoxelShape getSidesShape(BlockState state, BlockView view, BlockPos pos) {
        if (view instanceof RenderAttachedBlockView renderView) {
            Object data = renderView.getBlockEntityRenderAttachment(pos);
            if (data instanceof PartModelData partData) {
                return partData.sidesShape;
            }
        }

        if (view.getBlockEntity(pos) instanceof MultipartBlockEntity container) {
            return container.container.getSidesShape();
        }
        return super.getSidesShape(state, view, pos);
    }

    @Override
    public boolean hasDynamicCull(BlockState state) {
        return true;
    }

    @Override
    public void onStateReplaced(BlockState oldState, World world, BlockPos pos, BlockState newState, boolean bool) {
        if (newState.getBlock() != this) {
            BlockEntity be = world.getBlockEntity(pos);
            if (be instanceof MultipartBlockEntity) {
                MultipartBlockEntity container = (MultipartBlockEntity) be;
                container.onRemoved();
            }
        }
        super.onStateReplaced(oldState, world, pos, newState, bool);
    }

    @Override
    public void neighborUpdate(
        BlockState state, World world, BlockPos thisPos, Block otherBlock, BlockPos otherPos, boolean unknownBoolean
    ) {
        super.neighborUpdate(state, world, thisPos, otherBlock, otherPos, unknownBoolean);
        BlockEntity be = world.getBlockEntity(thisPos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity container = (MultipartBlockEntity) be;
            container.onNeighbourUpdate(otherPos);
        }
    }

    @Override
    public BlockState getStateForNeighborUpdate(
        BlockState state, Direction direction, BlockState neighborState, WorldAccess world, BlockPos pos,
        BlockPos neighborPos
    ) {
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity container = (MultipartBlockEntity) be;
            container.onNeighbourStateUpdate(direction, neighborPos, neighborState);
        }
        return super.getStateForNeighborUpdate(state, direction, neighborState, world, pos, neighborPos);
    }

    @Override
    protected ActionResult onUse(BlockState state, World world, BlockPos pos, PlayerEntity player, BlockHitResult hit) {
        ActionResult handled = ActionResult.PASS;
        TransientPartIdentifier target = getMultipartTarget(state, world, pos, hit.getPos());
        if (target != null) {
            handled = target.part.onUse(player, hit);
        }
        if (handled == ActionResult.PASS) {
            handled = super.onUse(state, world, pos, player, hit);
        }
        return handled;
    }

    @Override
    protected ItemActionResult onUseWithItem(
        ItemStack stack, BlockState state, World world, BlockPos pos, PlayerEntity player, Hand hand, BlockHitResult hit
    ) {
        ItemActionResult result = ItemActionResult.PASS_TO_DEFAULT_BLOCK_INTERACTION;
        TransientPartIdentifier target = getMultipartTarget(state, world, pos, hit.getPos());
        if (target != null) {
            result = target.part.onUseWithItem(stack, player, hand, hit);
        }
        if (result == ItemActionResult.PASS_TO_DEFAULT_BLOCK_INTERACTION) {
            result = super.onUseWithItem(stack, state, world, pos, player, hand, hit);
        }
        return result;
    }

    @Override
    public void onEntityCollision(BlockState state, World world, BlockPos pos, Entity entity) {
        super.onEntityCollision(state, world, pos, entity);
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity container = (MultipartBlockEntity) be;
            container.container.fireEvent(new PartEventEntityCollide(entity));
        }
    }

    @Override
    @Environment(EnvType.CLIENT)
    public ItemStack getPickStack(WorldView view, BlockPos pos, BlockState state) {
        MinecraftClient mc = MinecraftClient.getInstance();
        HitResult hit = mc.crosshairTarget;
        if (view != null && view == mc.world && hit != null && hit.getType() == HitResult.Type.BLOCK) {
            TransientPartIdentifier target = getMultipartTarget(state, view, pos, hit.getPos());
            if (target != null) {
                return target.part.getPickStack((BlockHitResult) hit);
            }
        }
        return ItemStack.EMPTY;
    }

    @Override
    public void randomTick(BlockState state, ServerWorld world, BlockPos pos, Random random) {
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity container) {
            container.container.fireEvent(PartRandomTickEvent.INSTANCE);
        }
    }

    @Override
    public void scheduledTick(BlockState state, ServerWorld world, BlockPos pos, Random random) {
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity container) {
            container.container.fireEvent(PartScheduledTickEvent.INSTANCE);
        }
    }

    @Override
    public void randomDisplayTick(BlockState state, World world, BlockPos pos, Random random) {
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity container) {
            container.container.fireEvent(PartRandomDisplayTickEvent.INSTANCE);
        }
    }

    @Override
    public void precipitationTick(BlockState state, World world, BlockPos pos, Biome.Precipitation precipitation) {
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity container) {
            container.container.fireEvent(new PartPrecipitationTickEvent(precipitation));
        }
    }

    // ###############
    //
    // Redstone
    //
    // ###############

    @Override
    public boolean emitsRedstonePower(BlockState state) {
        return state.get(EMITS_REDSTONE);
    }

    @Override
    public int getStrongRedstonePower(BlockState state, BlockView view, BlockPos pos, Direction oppositeFace) {
        BlockEntity be = view.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity multiBE) {
            return multiBE.container.getStrongRedstonePower(oppositeFace.getOpposite());
        }
        return 0;
    }

    @Override
    public int getWeakRedstonePower(BlockState state, BlockView view, BlockPos pos, Direction oppositeFace) {
        BlockEntity be = view.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity multiBE) {
            return multiBE.container.getWeakRedstonePower(oppositeFace.getOpposite());
        }
        return 0;
    }

    // ###############
    //
    // Waterloggable
    //
    // ###############

    @Override
    public boolean canFillWithFluid(@org.jetbrains.annotations.Nullable PlayerEntity player, BlockView world, BlockPos pos, BlockState state, Fluid fluid) {
        return testFluid(world, pos, fluid);
    }

    private static boolean testFluid(BlockView world, BlockPos pos, Fluid fluid) {
        if (fluid != Fluids.WATER) {
            return false;
        }
        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            return ((MultipartBlockEntity) be).container.properties.getValue(MultipartProperties.CAN_BE_WATERLOGGED);
        }
        return true;
    }

    @Override
    public boolean tryFillWithFluid(WorldAccess world, BlockPos pos, BlockState state, FluidState fluid) {
        if (!testFluid(world, pos, fluid.getFluid())) {
            return false;
        }
        return Waterloggable.super.tryFillWithFluid(world, pos, state, fluid);
    }

    @Override
    public FluidState getFluidState(BlockState state) {
        return state.get(Properties.WATERLOGGED) ? Fluids.WATER.getStill(false) : super.getFluidState(state);
    }

    // ###############
    //
    // IBlockCustomParticles
    //
    // ###############

    @Override
    @Environment(EnvType.CLIENT)
    public boolean spawnBreakingParticles(World world, BlockPos pos, BlockState state, Direction side, Vec3d hitVec) {
        TransientPartIdentifier target = getMultipartTarget(state, world, pos, hitVec);
        if (target != null) {

            if (target.extra instanceof IdSubPart<?>) {
                if (spawnSubBreakingParticles(hitVec, side, (IdSubPart<?>) target.extra)) {
                    return true;
                }
            } else if (target.extra instanceof IdAdditional) {
                for (AbstractPart extra : ((IdAdditional) target.extra).additional) {
                    extra.spawnHitParticle(side);
                }
            }

            return target.part.spawnHitParticle(side);
        }
        return false;
    }

    private static <Sub> boolean spawnSubBreakingParticles(Vec3d hitVec, Direction side, IdSubPart<Sub> extra) {
        return extra.part.spawnBreakingParticles(hitVec, extra.subpart, side);
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean spawnSprintingParticles(
        World world, BlockPos pos, BlockState state, Entity sprintingEntity,
        Random entityRandom
    ) {
        // Clearance always seems to be 0.2 for these entity-collision particles
        TransientPartIdentifier colliding =
            getMultipartColliding(state, world, pos, sprintingEntity.getBoundingBox(), 0.2);
        if (colliding != null) {
            return colliding.part.spawnSprintParticle(sprintingEntity, entityRandom);
        }

        // You pretty much always want to return true here, regardless of whether particles were actually spawned,
        // because returning true prevents the missing-texture particles from being spawned.
        return true;
    }

    @Override
    public boolean spawnFallParticles(
        ServerWorld world, BlockPos pos, BlockState state, LivingEntity fallenEntity,
        Random entityRandom
    ) {
        // Clearance always seems to be 0.2 for these entity-collision particles
        TransientPartIdentifier colliding =
            getMultipartColliding(state, world, pos, fallenEntity.getBoundingBox(), 0.2);
        if (colliding != null) {
            return colliding.part.onSpawnFallParticles(fallenEntity, entityRandom);
        }

        // You pretty much always want to return true here, regardless of whether particles were actually spawned,
        // because returning true prevents the missing-texture particles from being spawned.
        return true;
    }

    // ###############
    //
    // IBlockMultipart
    //
    // ###############

    @Override
    public Class<TransientPartIdentifier> getKeyClass() {
        return TransientPartIdentifier.class;
    }

    @Override
    public void onBlockBreakStart(
        BlockState state, World world, BlockPos pos, PlayerEntity player, TransientPartIdentifier subpart
    ) {
        onBlockBreakStart(state, world, pos, player);
    }

    @Override
    public float calcBlockBreakingDelta(
        BlockState state, PlayerEntity player, BlockView view, BlockPos pos, TransientPartIdentifier subpart
    ) {
        return subpart.part.calculateBreakingDelta(player);
    }

    @Override
    public void onBreak(
        World world, BlockPos pos, BlockState state, PlayerEntity player, TransientPartIdentifier subpart
    ) {
        if (subpart.extra instanceof IdSubPart<?>) {
            if (onSubpartBreak(player, (IdSubPart<?>) subpart.extra)) {
                return;
            }
        } else if (subpart.extra instanceof IdAdditional) {
            for (AbstractPart part : ((IdAdditional) subpart.extra).additional) {
                part.onBreak(player);
            }
        }
        if (!subpart.part.onBreak(player)) {
            onBreak(world, pos, state, player);
        }
    }

    private static <Sub> boolean onSubpartBreak(PlayerEntity player, IdSubPart<Sub> extra) {
        return extra.part.onSubpartBreak(player, extra.subpart);
    }

    @Override
    public boolean clearBlockState(World world, BlockPos pos, TransientPartIdentifier subpart) {
        if (subpart.extra instanceof IdSubPart<?>) {
            return clearSubPart((IdSubPart<?>) subpart.extra);
        }

        BlockEntity be = world.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity multi = (MultipartBlockEntity) be;
            return multi.container.removePart(subpart.part);
        }
        return world.removeBlock(pos, false);
    }

    private static <Sub> boolean clearSubPart(IdSubPart<Sub> extra) {
        return extra.part.clearSubpart(extra.subpart);
    }

    @Override
    public void onBroken(WorldAccess world, BlockPos pos, BlockState state, TransientPartIdentifier subpart) {
        onBroken(world, pos, state);
    }

    @Override
    public void afterBreak(
        World world, PlayerEntity player, BlockPos pos, BlockState state, BlockEntity blockEntity, ItemStack stack,
        TransientPartIdentifier subpart
    ) {
        if (subpart.extra instanceof IdSubPart<?>) {
            afterSubpartBreak(player, stack, (IdSubPart<?>) subpart.extra);
        } else {
            if (world instanceof ServerWorld sv) {
                LootContextParameterSet.Builder ctxBuilder = new LootContextParameterSet.Builder(sv);
                ctxBuilder.add(LootContextParameters.BLOCK_STATE, state);
                ctxBuilder.add(LootContextParameters.ORIGIN, Vec3d.ofCenter(pos));
                ctxBuilder.add(LootContextParameters.TOOL, stack);
                ctxBuilder.addOptional(LootContextParameters.THIS_ENTITY, player);
                ctxBuilder.addOptional(LootContextParameters.BLOCK_ENTITY, blockEntity);
                subpart.putLootContext(ctxBuilder);
                LootContextParameterSet context = ctxBuilder.build(PartLootParams.PART_TYPE);
                subpart.part.afterBreak(player);
                subpart.part.addDrops(createDropTarget(subpart.part), context);
                for (AbstractPart part : ((IdAdditional) subpart.extra).additional) {
                    part.afterBreak(player);
                    part.addDrops(createDropTarget(part), context);
                }
            } else {
                subpart.part.afterBreak(player);
                for (AbstractPart part : ((IdAdditional) subpart.extra).additional) {
                    part.afterBreak(player);
                }
            }
        }
    }

    private static <Sub> void afterSubpartBreak(PlayerEntity player, ItemStack tool, IdSubPart<Sub> extra) {
        extra.part.afterSubpartBreak(player, tool, extra.subpart);
    }

    static ItemDropTarget createDropTarget(AbstractPart part) {
        return new ItemDropTarget() {

            final boolean canDrop
                = part.holder.getContainer().getMultipartWorld().getGameRules().getBoolean(GameRules.DO_TILE_DROPS);

            Vec3d center = null;

            @Override
            public boolean dropsAsEntity() {
                return true;
            }

            @Override
            public void drop(ItemStack stack) {
                if (!canDrop) {
                    return;
                }

                if (center == null) {
                    center = part.getOutlineShape().getBoundingBox().getCenter();
                    center = center.add(Vec3d.of(part.holder.getContainer().getMultipartPos()));
                }
                drop(stack, center);
            }

            @Override
            public void drop(ItemStack stack, Vec3d pos) {
                if (!canDrop) {
                    return;
                }

                World world = part.holder.getContainer().getMultipartWorld();
                while (!stack.isEmpty()) {
                    ItemStack split = stack.split(world.random.nextInt(21) + 10);
                    ItemEntity ent = new ItemEntity(world, pos.x, pos.y, pos.z, split);
                    ent.setToDefaultPickupDelay();
                    ent.setVelocity(
                        world.random.nextGaussian() * 0.05, //
                        world.random.nextGaussian() * 0.05 + 0.2, //
                        world.random.nextGaussian() * 0.05//
                    );
                    world.spawnEntity(ent);
                }
            }

            @Override
            public void drop(ItemStack stack, Vec3d pos, Vec3d velocity) {
                if (!canDrop) {
                    return;
                }

                World world = part.holder.getContainer().getMultipartWorld();
                ItemEntity ent = new ItemEntity(world, pos.x, pos.y, pos.z, stack);
                ent.setVelocity(velocity);
                ent.setToDefaultPickupDelay();
                world.spawnEntity(ent);
            }
        };
    }

    @Override
    public List<ItemStack> getDroppedStacks(BlockState state, LootContextParameterSet.Builder builder) {
        DefaultedList<ItemStack> drops = DefaultedList.of();
        builder.add(LootContextParameters.BLOCK_STATE, state);
        BlockEntity be = builder.getOptional(LootContextParameters.BLOCK_ENTITY);

        LootContextParameterSet parameterSet = builder.build(LootContextTypes.BLOCK);

        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity mpbe = (MultipartBlockEntity) be;

            ItemDropTarget target = new ItemDropCollector(drops);

            for (PartHolder holder : mpbe.container.parts) {
                holder.part.addDrops(target, parameterSet);
            }
        }
        return drops;
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean playHitSound(
        World world, BlockPos pos, BlockState state, PlayerEntity player, TransientPartIdentifier subpart
    ) {
        if (subpart.extra instanceof IdSubPart<?>) {
            playSubpartHitSound(player, (IdSubPart<?>) subpart.extra);
        } else {
            subpart.part.playHitSound(player);
        }
        return true;
    }

    private static <T> void playSubpartHitSound(PlayerEntity player, IdSubPart<T> subpart) {
        subpart.part.playHitSound(player, subpart.subpart);
    }

    @Deprecated
    @Override
    public TransientPartIdentifier getTargetedMultipart(BlockState state, World world, BlockPos pos, Vec3d vec) {
        return getMultipartTarget(state, world, pos, vec);
    }

    @Override
    public TransientPartIdentifier getMultipartTarget(BlockState state, BlockView view, BlockPos pos, Vec3d vec) {
        BlockEntity be = view.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity multi = (MultipartBlockEntity) be;

            vec = vec.subtract(Vec3d.of(pos));

            for (PartHolder holder : multi.container.parts) {

                AbstractPart part = holder.getPart();

                if (part instanceof SubdividedPart<?>) {
                    TransientPartIdentifier id = getTargetSubPart((SubdividedPart<?>) part, vec);
                    if (id != null) {
                        return id;
                    }
                }

                if (doesContain(part, vec)) {
                    return new TransientPartIdentifier(part);
                }
            }
        }

        return null;
    }

    static boolean doesContain(AbstractPart part, Vec3d vec) {
        VoxelShape shape = part.getOutlineShape();
        for (Box box : shape.getBoundingBoxes()) {
            if (box.expand(0.01).contains(vec)) {
                return true;
            }
        }
        return false;
    }

    @Nullable
    private static <Sub> TransientPartIdentifier getTargetSubPart(SubdividedPart<Sub> part, Vec3d vec) {
        Sub subpart = part.getTargetedSubpart(vec);
        if (subpart != null) {
            return new TransientPartIdentifier(part, subpart);
        }
        return null;
    }

    @Nullable
    @Override
    public TransientPartIdentifier getMultipartColliding(
        BlockState state, BlockView view, BlockPos pos,
        Box toCollideWith, double clearance
    ) {
        BlockEntity be = view.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            MultipartBlockEntity multi = (MultipartBlockEntity) be;

            toCollideWith = toCollideWith.offset(Vec3d.of(pos).negate()).expand(clearance);

            for (PartHolder holder : multi.container.parts) {

                AbstractPart part = holder.getPart();

                // Presumably, sub-part collision shapes should be part of the parent-part collision shape

                if (isCollidingWith(part, toCollideWith)) {
                    return new TransientPartIdentifier(part);
                }
            }
        }

        return null;
    }

    static boolean isCollidingWith(AbstractPart part, Box toCollideWith) {
        VoxelShape shape = part.getCollisionShape();
        for (Box box : shape.getBoundingBoxes()) {
            if (box.intersects(toCollideWith)) {
                return true;
            }
        }
        return false;
    }

    @Override
    @Deprecated
    public VoxelShape getPartOutlineShape(BlockState state, World world, BlockPos pos, Vec3d hitVec) {
        return getPartOutline(state, world, pos, hitVec);
    }

    @Override
    public VoxelShape getPartOutline(BlockState state, BlockView world, BlockPos pos, Vec3d hitVec) {
        TransientPartIdentifier target = getMultipartTarget(state, world, pos, hitVec);
        if (target == null) {
            return VoxelShapes.empty();
        }
        float partialTicks = LibMultiPart.partialTickGetter.getAsFloat();
        if (target.extra instanceof IdSubPart<?>) {
            VoxelShape sub = getSubpartShape((IdSubPart<?>) target.extra, hitVec, partialTicks);
            if (sub != null) {
                return sub;
            }
        }
        return target.part.getDynamicShape(partialTicks, hitVec);
    }

    @Nullable
    private static <S> VoxelShape getSubpartShape(IdSubPart<S> sub, Vec3d vec, float partialTicks) {
        return sub.part.getSubpartDynamicShape(vec, sub.subpart, partialTicks);
    }
}
