/*
 * Copyright (c) 2019 AlexIIL
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */
package alexiil.mc.lib.multipart.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import net.minecraft.block.Block;
import net.minecraft.block.BlockState;
import net.minecraft.block.entity.BlockEntity;
import net.minecraft.fluid.Fluid;
import net.minecraft.fluid.Fluids;
import net.minecraft.state.property.Properties;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.DirectionTransformation;
import net.minecraft.util.shape.VoxelShape;
import net.minecraft.world.BlockView;
import net.minecraft.world.World;

import alexiil.mc.lib.multipart.api.MultipartContainer;
import alexiil.mc.lib.multipart.api.MultipartContainer.MultipartCreator;
import alexiil.mc.lib.multipart.api.MultipartContainer.PartOffer;
import alexiil.mc.lib.multipart.api.MultipartHolder;
import alexiil.mc.lib.multipart.api.MultipartUtil;
import alexiil.mc.lib.multipart.api.NativeMultipart;

/** Contains the backend for {@link MultipartUtil}. */
public final class MultipartUtilImpl {

    @Nullable
    public static PartContainer get(World world, BlockPos pos) {
        return get((BlockView) world, pos);
    }

    @Nullable
    public static PartContainer get(BlockView view, BlockPos pos) {
        BlockEntity be = view.getBlockEntity(pos);
        if (be instanceof MultipartBlockEntity) {
            return ((MultipartBlockEntity) be).container;
        }
        return null;
    }

    @Nullable
    public static MultipartContainer turnIntoMultipart(World world, BlockPos pos) {
        // See if there's an existing multipart that doesn't need to be changed
        PartContainer existing = get(world, pos);
        if (existing != null) {
            return existing;
        }

        Fluid fluid = world.getFluidState(pos).getFluid();
        boolean hasWater = fluid == Fluids.WATER;
        if (fluid != Fluids.WATER && fluid != Fluids.EMPTY) {
            return null;
        }

        BlockState state = world.getBlockState(pos);
        NativeMultipart nativeBlock = getNative(world, pos);
        if (nativeBlock != null) {
            List<MultipartCreator> conversions = nativeBlock.getMultipartConversion(world, pos, state);
            if (conversions == null) {
                return null;
            }
            if (!conversions.isEmpty()) {
                PartOffer offer = offerAdder(world, pos, hasWater, conversions, null, false);
                if (offer == null) {
                    return null;
                }
                offer.apply();
                return offer.getHolder().getContainer();
            }
        }
        return null;
    }

    private static NativeMultipart getNative(World world, BlockPos pos) {
        BlockState state = world.getBlockState(pos);
        if (state.getBlock() instanceof NativeMultipart nativeBlock) {
            return nativeBlock;
        }
        return NativeMultipart.LOOKUP.find(world, pos, null);
    }

    @Nullable
    public static PartOffer offerNewPart(
        World world, BlockPos pos, MultipartCreator creator, boolean respectEntityBBs
    ) {
        // See if there's an existing multipart that we can add to
        PartContainer currentContainer = get(world, pos);
        if (currentContainer != null) {
            return currentContainer.offerNewPart(creator, respectEntityBBs);
        }

        Fluid fluid = world.getFluidState(pos).getFluid();
        boolean hasWater = fluid == Fluids.WATER;
        if (fluid != Fluids.WATER && fluid != Fluids.EMPTY) {
            return null;
        }

        BlockState state = world.getBlockState(pos);
        NativeMultipart nativeBlock = getNative(world, pos);
        if (nativeBlock != null) {
            List<MultipartCreator> conversions = nativeBlock.getMultipartConversion(world, pos, state);
            if (conversions == null) {
                return null;
            }
            if (!conversions.isEmpty()) {
                return offerAdder(world, pos, hasWater, conversions, creator, respectEntityBBs);
            }
        } else if (!state.isReplaceable()) {
            return null;
        }

        MultipartBlockEntity be = new MultipartBlockEntity(pos, state);
        be.setWorld(world);
        PartContainer container = new PartContainer(be, DirectionTransformation.IDENTITY);
        PartHolder holder = new PartHolder(container, creator);
        VoxelShape shape = holder.part.getCollisionShape();

        if (respectEntityBBs && !shape.isEmpty()) {
            if (!world.doesNotIntersectEntities(null, shape.offset(pos.getX(), pos.getY(), pos.getZ()))) {
                return null;
            }
        }
        return new PartOffer() {
            @Override
            public MultipartHolder getHolder() {
                return holder;
            }

            @Override
            public void apply() {
                // Actually place the new multipart
                BlockState newState = LibMultiPart.BLOCK.getDefaultState();
                newState = newState.with(Properties.WATERLOGGED, hasWater);
                world.setBlockState(pos, newState, Block.NOTIFY_ALL);
                MultipartBlockEntity newBe = (MultipartBlockEntity) world.getBlockEntity(pos);
                assert newBe != null;
                newBe.container = container;
                container.blockEntity = newBe;
                container.addPartInternal(holder);
            }
        };
    }

    @Nullable
    private static PartOffer offerAdder(
        World world, BlockPos pos, boolean hasWater, List<MultipartCreator> existing, MultipartCreator creatorB,
        boolean respectEntityBBs
    ) {
        MultipartBlockEntity be = new MultipartBlockEntity(
            pos, LibMultiPart.BLOCK.getDefaultState().with(Properties.WATERLOGGED, hasWater)
        );
        be.setWorld(world);
        PartContainer container = new PartContainer(be, DirectionTransformation.IDENTITY);

        List<PartHolder> existingHolders = new ArrayList<>();
        for (MultipartCreator creator : existing) {
            PartHolder holder = new PartHolder(container, creator);
            existingHolders.add(holder);

            // Add the existing ones so that they can intercept the offered part
            container.parts.add(holder);

            VoxelShape shape = holder.part.getCollisionShape();
            if (respectEntityBBs && !shape.isEmpty()) {
                if (!world.doesNotIntersectEntities(null, shape.offset(pos.getX(), pos.getY(), pos.getZ()))) {
                    return null;
                }
            }
        }

        PartHolder offeredHolder = creatorB == null ? null : new PartHolder(container, creatorB);

        if (offeredHolder != null) {
            VoxelShape shape = offeredHolder.part.getCollisionShape();
            if (respectEntityBBs && !shape.isEmpty()) {
                if (!world.doesNotIntersectEntities(null, shape.offset(pos.getX(), pos.getY(), pos.getZ()))) {
                    return null;
                }
            }

            container.recalculateShape();

            if (!container.canAdd(offeredHolder, true)) {
                return null;
            }
        }

        if (offeredHolder == null && existingHolders.isEmpty()) {
            return null;
        }

        return new PartOffer() {
            @Override
            public MultipartHolder getHolder() {
                return offeredHolder != null ? offeredHolder : existingHolders.get(0);
            }

            @Override
            public void apply() {
                // Cleanup the temporary additions
                container.parts.clear();

                // Inform the new parts that they are about to replace the existing NativeMultipart
                for (PartHolder holder : existingHolders) {
                    holder.part.preConvertNativeMultipart();
                }

                // Get the old block state for update notifications
                BlockState oldState = world.getBlockState(pos);

                // Actually place the new multipart
                BlockState newState = LibMultiPart.BLOCK.getDefaultState();
                newState = newState.with(Properties.WATERLOGGED, hasWater);
                world.setBlockState(pos, newState, 0); // do not notify yet
                MultipartBlockEntity newBe = (MultipartBlockEntity) world.getBlockEntity(pos);
                assert newBe != null;
                newBe.container = container;
                container.blockEntity = newBe;
                for (PartHolder holder : existingHolders) {
                    container.addPartInternal(holder);
                }
                if (offeredHolder != null) {
                    container.addPartInternal(offeredHolder);
                }

                // Notify now that the block entity is set up
                // FIXME: for some reason, this still isn't updating blocks the way a normal setBlockState with NOTIFY_ALL would.
                world.updateListeners(pos, oldState, newState, Block.NOTIFY_ALL);
                world.updateNeighbors(pos, LibMultiPart.BLOCK);
                oldState.prepare(world, pos, Block.NOTIFY_LISTENERS);
                newState.updateNeighbors(world, pos, Block.NOTIFY_LISTENERS);
                newState.prepare(world, pos, Block.NOTIFY_LISTENERS);
            }
        };
    }
}
