package io.github.cottonmc.cotton.gui.widget;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1921;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_437;
import net.minecraft.class_6381;
import net.minecraft.class_6382;
import io.github.cottonmc.cotton.gui.client.ScreenDrawing;
import io.github.cottonmc.cotton.gui.impl.client.NarrationMessages;
import io.github.cottonmc.cotton.gui.impl.mixin.client.TextFieldWidgetAccessor;
import io.github.cottonmc.cotton.gui.widget.data.InputResult;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.glfw.GLFW;

import java.util.function.Consumer;
import java.util.function.Predicate;

public class WTextField extends WWidget {
	public static final int TEXT_PADDING_X = 4;
	public static final int TEXT_PADDING_Y = 6;
	public static final int CURSOR_PADDING_Y = 4;
	public static final int CURSOR_HEIGHT = 12;

	@Environment(EnvType.CLIENT)
	@Nullable
	private class_327 textRenderer;

	private String text = "";
	private int maxLength = 16;
	private boolean editable = true;
	private int tickCount = 0;

	private int disabledColor = 0x707070;
	private int enabledColor = 0xE0E0E0;
	private int suggestionColor = 0x808080;

	private static final int CURSOR_COLOR = 0xFFD0D0D0;

	@Nullable
	private class_2561 suggestion = null;

	// Index of the leftmost character to be rendered.
	private int scrollOffset = 0;

	private int cursor = 0;
	/**
	 * If not -1, select is the "anchor point" of a selection. That is, if you hit shift+left with no existing
	 * selection, the selection will be anchored to where you were, but the cursor will move left, expanding the
	 * selection as you continue to move left. If you move to the right, eventually you'll overtake the anchor, drop the
	 * anchor at the same place and start expanding the selection rightwards instead.
	 */
	private int select = -1;

	private Consumer<String> onChanged;

	private Predicate<String> textPredicate;

	public WTextField() {
	}

	public WTextField(class_2561 suggestion) {
		this.suggestion = suggestion;
	}

	/**
	 * Sets the text of this text field.
	 * If the text is more than the {@linkplain #getMaxLength() max length},
	 * it'll be shortened to the max length.
	 *
	 * @param s the new text
	 */
	public void setText(String s) {
		setTextWithResult(s);
	}

	private boolean setTextWithResult(String s) {
		if (this.textPredicate == null || this.textPredicate.test(s)) {
			this.text = (s.length() > maxLength) ? s.substring(0, maxLength) : s;
			// Call change listener
			if (onChanged != null) onChanged.accept(this.text);
			// Reset cursor if needed
			cursor = clampCursor(cursor);
			return true;
		}

		return false;
	}

	/**
	 * {@return the text in this text field}
	 */
	public String getText() {
		return this.text;
	}

	@Override
	public boolean canResize() {
		return true;
	}

	@Override
	public void tick() {
		super.tick();
		this.tickCount++;
	}

	@Override
	public void setSize(int x, int y) {
		super.setSize(x, 20);
	}

	private int clampCursor(int cursor) {
		return class_3532.method_15340(cursor, 0, text.length());
	}

	public void setCursorPos(int location) {
		this.cursor = clampCursor(location);
		scrollCursorIntoView();
	}

	public int getMaxLength() {
		return this.maxLength;
	}

	public int getCursor() {
		return this.cursor;
	}

	@Environment(EnvType.CLIENT)
	private class_327 getTextRenderer() {
		return textRenderer != null ? textRenderer : (textRenderer = class_310.method_1551().field_1772);
	}

	@Environment(EnvType.CLIENT)
	public void scrollCursorIntoView() {
		class_327 font = getTextRenderer();

		if (scrollOffset > cursor) {
			scrollOffset = cursor;
		}
		if (scrollOffset < cursor && font.method_27523(text.substring(scrollOffset), width - TEXT_PADDING_X * 2).length() + scrollOffset < cursor) {
			scrollOffset = cursor;
		}

		checkScrollOffset();
	}

	@Environment(EnvType.CLIENT)
	private void checkScrollOffset() {
		int rightMostScrollOffset = text.length() - getTextRenderer().method_27524(text, width - TEXT_PADDING_X * 2, true).length();
		scrollOffset = Math.min(rightMostScrollOffset, scrollOffset);
	}

	@Nullable
	public String getSelection() {
		if (select < 0) return null;
		if (select == cursor) return null;

		//Tidy some things
		if (select > text.length()) select = text.length();
		cursor = clampCursor(cursor);

		int start = Math.min(select, cursor);
		int end = Math.max(select, cursor);

		return text.substring(start, end);
	}

	public boolean isEditable() {
		return this.editable;
	}

	@Environment(EnvType.CLIENT)
	protected void renderBox(class_332 context, int x, int y) {
		var texture = TextFieldWidgetAccessor.libgui$getTextures().method_52729(isEditable(), isFocused());
		context.method_52706(class_1921::method_62277, texture, x - 1, y - 1, width + 2, height + 2);
	}

	@Environment(EnvType.CLIENT)
	protected void renderText(class_332 context, int x, int y, String visibleText) {
		int textColor = this.editable ? this.enabledColor : this.disabledColor;
		context.method_51433(getTextRenderer(), visibleText, x + TEXT_PADDING_X, y + TEXT_PADDING_Y, textColor, true);
	}

	@Environment(EnvType.CLIENT)
	protected void renderCursor(class_332 context, int x, int y, String visibleText) {
		if (this.tickCount / 6 % 2 == 0) return;
		if (this.cursor < this.scrollOffset) return;
		if (this.cursor > this.scrollOffset + visibleText.length()) return;
		int cursorOffset = getTextRenderer().method_1727(visibleText.substring(0, this.cursor - this.scrollOffset));
		ScreenDrawing.coloredRect(context, x + TEXT_PADDING_X + cursorOffset, y + CURSOR_PADDING_Y, 1, CURSOR_HEIGHT, CURSOR_COLOR);
	}

	@Environment(EnvType.CLIENT)
	protected void renderSuggestion(class_332 context, int x, int y) {
		if (this.suggestion == null) return;
		context.method_51439(getTextRenderer(), suggestion, x + TEXT_PADDING_X, y + TEXT_PADDING_Y, this.suggestionColor, true);
	}

	@Environment(EnvType.CLIENT)
	protected void renderSelection(class_332 context, int x, int y, String visibleText) {
		if (select == cursor || select == -1) return;

		int textLength = visibleText.length();

		int left = Math.min(cursor, select);
		int right = Math.max(cursor, select);

		if (right < scrollOffset || left > scrollOffset + textLength) return;

		int normalizedLeft = Math.max(scrollOffset, left) - scrollOffset;
		int normalizedRight = Math.min(scrollOffset + textLength, right) - scrollOffset;

		class_327 font = getTextRenderer();
		int leftCaret = font.method_1727(visibleText.substring(0, normalizedLeft));
		int selectionWidth = font.method_1727(visibleText.substring(normalizedLeft, normalizedRight));

		drawHighlight(context, x + TEXT_PADDING_X + leftCaret, y + CURSOR_PADDING_Y, selectionWidth, CURSOR_HEIGHT);
	}

	@Environment(EnvType.CLIENT)
	protected void renderTextField(class_332 context, int x, int y) {
		checkScrollOffset();
		String visibleText = getTextRenderer().method_27523(this.text.substring(this.scrollOffset), this.width - 2 * TEXT_PADDING_X);
		renderBox(context, x, y);
		renderText(context, x, y, visibleText);
		if (this.text.isEmpty() && !this.isFocused()) {
			renderSuggestion(context, x, y);
		}
		if (this.isFocused()) {
			renderCursor(context, x, y, visibleText);
		}
		renderSelection(context, x, y, visibleText);
	}

	@Environment(EnvType.CLIENT)
	private void drawHighlight(class_332 context, int x, int y, int width, int height) {
		context.method_51739(class_1921.method_51786(), x, y, x + width, y + height, 0xFF_0000FF);
	}

	public WTextField setTextPredicate(Predicate<String> predicate_1) {
		this.textPredicate = predicate_1;
		return this;
	}

	public WTextField setChangedListener(Consumer<String> listener) {
		this.onChanged = listener;
		return this;
	}

	public WTextField setMaxLength(int max) {
		this.maxLength = max;
		if (this.text.length() > max) {
			setText(this.text.substring(0, max));
		}
		return this;
	}

	public WTextField setEnabledColor(int col) {
		this.enabledColor = col;
		return this;
	}

	public WTextField setSuggestionColor(int suggestionColor) {
		this.suggestionColor = suggestionColor;
		return this;
	}

	public WTextField setDisabledColor(int col) {
		this.disabledColor = col;
		return this;
	}

	public WTextField setEditable(boolean editable) {
		this.editable = editable;
		return this;
	}

	@Nullable
	public class_2561 getSuggestion() {
		return suggestion;
	}

	public WTextField setSuggestion(@Nullable class_2561 suggestion) {
		this.suggestion = suggestion;
		return this;
	}

	public boolean canFocus() {
		return true;
	}

	@Override
	public void onFocusGained() {
	}

	@Environment(EnvType.CLIENT)
	@Override
	public void paint(class_332 context, int x, int y, int mouseX, int mouseY) {
		renderTextField(context, x, y);
	}

	@Environment(EnvType.CLIENT)
	@Override
	public InputResult onClick(int x, int y, int button) {
		requestFocus();
		cursor = getCaretPosition(x - TEXT_PADDING_X);
		scrollCursorIntoView();
		return InputResult.PROCESSED;
	}

	@Environment(EnvType.CLIENT)
	public int getCaretPosition(int clickX) {
		if (clickX < 0) return 0;
		int lastPos = 0;
		checkScrollOffset();
		String string = text.substring(scrollOffset);
		class_327 font = getTextRenderer();
		for (int i = 0; i < string.length(); i++) {
			int w = font.method_1727(string.charAt(i) + "");
			if (lastPos + w >= clickX) {
				if (clickX - lastPos < w / 2) {
					return i + scrollOffset;
				}
			}
			lastPos += w;
		}
		return string.length();
	}

	@Environment(EnvType.CLIENT)
	@Override
	public InputResult onCharTyped(char ch) {
		if (!isEditable()) return InputResult.IGNORED;
		insertText(ch + "");
		return InputResult.PROCESSED;
	}

	@Environment(EnvType.CLIENT)
	private void insertText(String toInsert) {
		String before, after;
		if (select != -1 && select != cursor) {
			int left = Math.min(cursor, select);
			int right = Math.max(cursor, select);
			before = this.text.substring(0, left);
			after = this.text.substring(right);
		} else {
			before = this.text.substring(0, cursor);
			after = this.text.substring(cursor);
		}
		if (before.length() + after.length() + toInsert.length() > maxLength) return;
		if (setTextWithResult(before + toInsert + after)) {
			select = -1;
			cursor = (before + toInsert).length();
			scrollCursorIntoView();
		}
	}

	@Environment(EnvType.CLIENT)
	private void copySelection() {
		String selection = getSelection();
		if (selection != null) {
			class_310.method_1551().field_1774.method_1455(selection);
		}
	}

	@Environment(EnvType.CLIENT)
	private void paste() {
		String clip = class_310.method_1551().field_1774.method_1460();
		insertText(clip);
	}

	@Environment(EnvType.CLIENT)
	private void deleteSelection() {
		int left = Math.min(cursor, select);
		int right = Math.max(cursor, select);
		if (setTextWithResult(text.substring(0, left) + text.substring(right))) {
			select = -1;
			cursor = clampCursor(left);
			scrollCursorIntoView();
		}
	}

	@Environment(EnvType.CLIENT)
	private void delete(int modifiers, boolean backwards) {
		if (select == -1 || select == cursor) {
			select = skipCharacters((GLFW.GLFW_MOD_CONTROL & modifiers) != 0, backwards ? -1 : 1);
		}
		deleteSelection();
	}

	@Environment(EnvType.CLIENT)
	private int skipCharacters(boolean skipMany, int direction) {
		if (direction != -1 && direction != 1) return cursor;
		int position = cursor;
		while (true) {
			position += direction;
			if (position < 0) {
				return 0;
			}
			if (position > text.length()) {
				return text.length();
			}
			if (!skipMany) return position;
			if (position < text.length() && Character.isWhitespace(text.charAt(position))) {
				return position;
			}
		}
	}

	@Environment(EnvType.CLIENT)
	public void onDirectionalKey(int direction, int modifiers) {
		if ((GLFW.GLFW_MOD_SHIFT & modifiers) != 0) {
			if (select == -1 || select == cursor) select = cursor;
			cursor = skipCharacters((GLFW.GLFW_MOD_CONTROL & modifiers) != 0, direction);
		} else {
			if (select != -1) {
				cursor = direction < 0 ? Math.min(cursor, select) : Math.max(cursor, select);
				select = -1;
			} else {
				cursor = skipCharacters((GLFW.GLFW_MOD_CONTROL & modifiers) != 0, direction);
			}
		}
	}

	@Environment(EnvType.CLIENT)
	@Override
	public InputResult onKeyPressed(int ch, int key, int modifiers) {
		if (!isEditable()) return InputResult.IGNORED;

		if (class_437.method_25438(ch)) {
			copySelection();
			return InputResult.PROCESSED;
		} else if (class_437.method_25437(ch)) {
			paste();
			return InputResult.PROCESSED;
		} else if (class_437.method_25439(ch)) {
			select = 0;
			cursor = text.length();
			return InputResult.PROCESSED;
		}

		switch (ch) {
			case GLFW.GLFW_KEY_DELETE -> delete(modifiers, false);
			case GLFW.GLFW_KEY_BACKSPACE -> delete(modifiers, true);
			case GLFW.GLFW_KEY_LEFT -> onDirectionalKey(-1, modifiers);
			case GLFW.GLFW_KEY_RIGHT -> onDirectionalKey(1, modifiers);
			case GLFW.GLFW_KEY_HOME, GLFW.GLFW_KEY_UP -> {
				if ((GLFW.GLFW_MOD_SHIFT & modifiers) == 0) {
					select = -1;
				}
				cursor = 0;
			}
			case GLFW.GLFW_KEY_END, GLFW.GLFW_KEY_DOWN -> {
				if ((GLFW.GLFW_MOD_SHIFT & modifiers) == 0) {
					select = -1;
				}
				cursor = text.length();
			}
			default -> {
				return InputResult.IGNORED;
			}
		}
		scrollCursorIntoView();

		return InputResult.PROCESSED;
	}

	@Override
	public void addNarrations(class_6382 builder) {
		builder.method_37034(class_6381.field_33788, class_2561.method_43469(NarrationMessages.TEXT_FIELD_TITLE_KEY, text));

		if (suggestion != null) {
			builder.method_37034(class_6381.field_33790, class_2561.method_43469(NarrationMessages.TEXT_FIELD_SUGGESTION_KEY, suggestion));
		}
	}
}
